-- Create the protocol
local txse_grass = Proto("txse_grass", "TXSE Grass UDP Protocol")

-- Packet type value map
local packet_type_names = {
	[0] = "INVALID",
	[1] = "START_OF_STREAM",
	[2] = "HEARTBEAT",
	[3] = "NEW_GRASS_DATA",
	[4] = "GAPFILL_REQUEST",
	[5] = "GAPFILL_RESPONSE",
	[255] = "END_OF_STREAM",
}

-- Define protocol fields
local f_session_id = ProtoField.uint64("txse_grass.session_id", "Session ID", base.DEC)
local f_payload_offset = ProtoField.uint64("txse_grass.payload_starting_offset", "Payload Starting Offset", base.DEC)
local f_payload_length = ProtoField.uint16("txse_grass.trailing_payload_length", "Trailing Payload Length", base.DEC)
local f_stream_id = ProtoField.uint8("txse_grass.stream_id", "Stream ID", base.DEC)
local f_packet_type = ProtoField.uint8("txse_grass.type", "Packet Type", base.DEC, packet_type_names)
local f_payload = ProtoField.bytes("txse_grass.payload", "Payload")

-- Register fields
txse_grass.fields = {
	f_session_id,
	f_payload_offset,
	f_payload_length,
	f_stream_id,
	f_packet_type,
	f_payload
}

function txse_grass.dissector(buffer, pinfo, tree)
	local length = buffer:len()
	if length < 20 then
		-- Not enough data for a PacketHeader (20 bytes)
		return 0
	end

	pinfo.cols.protocol = txse_grass.name

	-- Parse the header fields
	local session_id = buffer(0, 8):le_uint64()
	local payload_offset = buffer(8, 8):le_uint64()
	local payload_length = buffer(16, 2):le_uint()
	local stream_id = buffer(18, 1):uint()
	local packet_type = buffer(19, 1):uint()

	-- Get packet type name
	local packet_type_name = packet_type_names[packet_type] or "UNKNOWN"

	-- Set info column (convert uint64 to string for formatting)
	pinfo.cols.info = string.format("Type: %s, Session: %s, Stream: %d, Offset: %s, Length: %d",
		packet_type_name, tostring(session_id), stream_id, tostring(payload_offset), payload_length)

	-- Create protocol tree
	local subtree = tree:add(txse_grass, buffer(), string.format("TXSE Grass %s", packet_type_name))

	-- Add header fields
	subtree:add_le(f_session_id, buffer(0, 8))
	subtree:add_le(f_payload_offset, buffer(8, 8))
	subtree:add_le(f_payload_length, buffer(16, 2))
	subtree:add(f_stream_id, buffer(18, 1))
	subtree:add(f_packet_type, buffer(19, 1))

	-- Add payload if present
	if length > 20 then
		local actual_payload_length = length - 20
		local payload_item = subtree:add(f_payload, buffer(20, actual_payload_length))

		-- If payload is truncated, add ellipsis to indicate more data
		if actual_payload_length < payload_length then
			payload_item:append_text(string.format(" ... (%d of %d bytes captured)",
				actual_payload_length, payload_length))
		end
	end

	-- Validate session_id is not 0 (invalid)
	if tostring(session_id) == "0" then
		subtree:add_expert_info(PI_MALFORMED, PI_WARN, "Invalid session_id (0)")
	end

	return length
end

-- Register for "Decode As..." on UDP ports
-- Use with: tshark -d udp.port==<port>,txse_grass
-- Or for all ports: tshark -d udp.port==0-65535,txse_grass
DissectorTable.get("udp.port"):add_for_decode_as(txse_grass)
