local n_bits = {
    0x1,
    0x3,
    0x7,
    0xf,
    0x1f,
    0x3f,
    0x7f,
    0xff,
    0x1ff,
    0x3ff,
    0x7ff,
    0xfff,
    0x1fff,
    0x3fff,
    0x7fff,
    0xffff,
}

local extract_bits = function(number, offset, length_arg)
    local length = length_arg or 1
    if offset < 0 then
        error("offset cannot be negative")
    end
    if length <= 0 then
        error("length cannot be negatve or zero")
    end
    if offset >= 31 then
        error("offset cannot exceed 31")
    end
    if length > 32 then
        error("length cannot exceed 32")
    end
    if offset + length > 32 then
        error("32-bit limitation on offset + length")
    end
    return bit.band(bit.rshift(number, offset), n_bits[length])
end

local ProtoFieldCache = {
    new = function()
        return {
            entries = {},
            get = function(self, field_spec)
                local key = ""
                for k, v in pairs(field_spec) do
                    key = key .. "|" .. tostring(k) .. "@" .. tostring(v) .. "^"
                end
                local entry = self.entries[key]
                if not entry then
                    entry = ProtoField.new(field_spec.name, field_spec.abbr, field_spec.ftype, field_spec.value_map)
                    self.entries[key] = entry
                end
                return entry
            end,
        }
    end
}

local Field =
{
    new = function(spec, proto_field_cache, field_name)
        local field_spec = spec.fields[field_name] or error("Did not find field: " .. field_name)
        local proto_field = proto_field_cache:get(field_spec)
        return {
            name = field_name,
            is_presence_field = field_spec.is_presence_field or false,
            parse = function(self, message_buffer, pinfo, subtree, offset)
                if offset >= message_buffer:len() then
                    subtree:add_packet_field(proto_field):add_text(
                        string.format(" *** Runt message missing entire field expected at offset:%d ***", offset)
                    )
                elseif (offset + (field_spec.length or 0)) > message_buffer:len() then
                    subtree:add_packet_field(proto_field, message_buffer(offset), field_spec.encoding):append_text(
                        string.format(
                            " *** Runt missing missing part of field starting at offset:%d for length %d ***",
                            offset,
                            field_spec.length
                        )
                    )
                else
                    local field_buffer = message_buffer(offset, field_spec.length)
                    local tree_item = subtree:add_packet_field(proto_field, field_buffer, field_spec.encoding)
                    if field_spec.custom_decode then
                        field_spec.custom_decode(field_buffer, pinfo, tree_item)
                    end
                end
            end,
            length = field_spec.length,
        }
    end,
}

local Message = {
    new = function(spec, proto_field_cache, msg_def)
        local fields = {}
        local presence_fields = {}
        local min_size = 0
        for _, field_name in ipairs(msg_def.fields) do
            local field = Field.new(spec, proto_field_cache, field_name)
            table.insert(fields, field)
            min_size = min_size + (field.length or 0)
        end
        local bit = 0
        for _, field_name in ipairs(msg_def.presence_fields or {}) do
            local field = Field.new(spec, proto_field_cache, field_name)
            table.insert(presence_fields, { field = field, bit = bit })
            bit = bit + 1
        end
        return {
            name = msg_def.name,
            type = string.byte(msg_def.type),
            parse = function(self, message_buffer, pinfo, subtree)
                local current_offset = 0
                local presence_bits = 0
                for _, field in ipairs(fields) do
                    local length = field.length or 0
                    field:parse(message_buffer, pinfo, subtree, current_offset)
                    if field.is_presence_field then
                        presence_bits = message_buffer(current_offset, length):le_uint()
                    end
                    current_offset = current_offset + length
                end
                for _, field_descriptor in ipairs(presence_fields) do
                    if extract_bits(presence_bits, field_descriptor.bit) == 1 then
                        local field = field_descriptor.field
                        local length = field.length or 0
                        field:parse(message_buffer, pinfo, subtree, current_offset)
                        current_offset = current_offset + length
                    end
                end
            end,
            min_size = min_size,
        }
    end,
}

local Messages = {
    new = function(spec)
        local proto_field_cache = ProtoFieldCache.new()
        local messages = {}
        local mapping = {}
        for _, msg_def in ipairs(spec.messages) do
            local message = Message.new(spec, proto_field_cache, msg_def)
            table.insert(messages, message)
            mapping[message.type] = message
        end
        local proto_fields = {}
        for _, field in pairs(proto_field_cache.entries) do
            table.insert(proto_fields, field)
        end
        return {
            messages = messages,
            mapping = mapping,
            proto_fields = proto_fields,
        }
    end,
}

local TxseProto = {
    new = function(spec)
        local proto = Proto.new(spec.name, spec.description)
        local messages = Messages.new(spec)
        proto.fields = messages.proto_fields
        return proto, messages
    end,
}

return {
    Proto = TxseProto,
    extract_bits = extract_bits,
}
