--[[

Resources:

- Example dissector: https://www.wireshark.org/docs/wsdg_html_chunked/wslua_dissector_example.html

- desegment information under "TCP reassembly": https://wiki.wireshark.org/Lua/Dissectors
  - Explains some of the loop and pinfo complexity in the dissector

- "register_heuristic" - https://www.wireshark.org/docs/wsdg_html_chunked/lua_module_Proto.html

- Header file for ftypes: https://www.wireshark.org/docs/wsar_html/ftypes_8h_source.html

- List of ENC_* values: https://www.wireshark.org/docs/wsar_html/group__prototree.html

- Install by copying or symlinking *.lua into ~/.local/lib/wireshark/plugins
- Refresh wireshark with the Analyze->Reload Lua Plugins action
- Force RAKE TCP decoder with Analyze->Decode As...
- Heuristic will auto decode "if it makes sense"
- tshark examples:

```
$ tshark -r rake.pcap 'tcp.len>0'
    8   0.045328 192.168.102.102 → 10.18.0.23   TXSE_RAKE_TCP 87 LogonRequest
    9   0.046031 192.168.102.102 → 10.18.0.23   TXSE_RAKE_TCP 87 LogonRequest
   12   0.051827 192.168.102.102 → 10.18.0.24   TXSE_RAKE_TCP 87 LogonRequest
   15   0.095874   10.18.0.23 → 192.168.102.102 TXSE_RAKE_TCP 89 LogonResponse
   17   0.097364   10.18.0.23 → 192.168.102.102 TXSE_RAKE_TCP 89 LogonResponse
```

```
$ tshark -r rake.pcap -O txse_rake_tcp 'frame.number==15'
Frame 15: 89 bytes on wire (712 bits), 89 bytes captured (712 bits) on interface utun4, id 0 (inbound)
Null/Loopback
Internet Protocol Version 4, Src: 10.18.0.23, Dst: 192.168.102.102
Transmission Control Protocol, Src Port: 18111, Dst Port: 53868, Seq: 1, Ack: 32, Len: 33
TXSE RAKE TCP LogonResponse
    Message Length: 31
    Message Type: 'A'
    Session: 4918266356343464677
        Environment: DA11
        Timestamp: Aug 12, 2025 10:51:01.000000000 CDT
    Next Sequence Number: 1
    Highest Known Sequence: 0
    Response Code: SUCCESS (0)
    Number of Stream IDs: 0
    Instance: 0
```

]]

local TXSE = require("txse-lib")

local cached_txse_seed_unseq = nil
local txse_seed_unseq = function()
	if cached_txse_seed_unseq == nil then
		cached_txse_seed_unseq = Dissector.get("txse_seed_unseq") or error("Unable to load txse_seed_unseq dissector")
	end
	return cached_txse_seed_unseq
end

local cached_txse_seed_seq = nil
local txse_seed_seq = function()
	if cached_txse_seed_seq == nil then
		cached_txse_seed_seq = Dissector.get("txse_seed_seq") or error("Unable to load txse_seed_seq dissector")
	end
	return cached_txse_seed_seq
end

local RakeTcpSpec = {
	name = "txse_rake_tcp",
	description = "TXSE RAKE TCP Session Framing Protocol v0.6",
	fields = {
		msg_length = {
			length = 2,
			abbr = "txse_rake_tcp.msg_length",
			name = "Message Length",
			ftype = ftypes.INT16,
			encoding = ENC_LITTLE_ENDIAN,
		},
		msg_type = {
			length = 1,
			abbr = "txse_rake_tcp.msg_type",
			name = "Message Type",
			ftype = ftypes.CHAR,
			encoding = ENC_ASCII,
		},
		rake_session = {
			length = 8,
			abbr = "txse_rake_tcp.session",
			name = "Session",
			ftype = ftypes.INT64,
			encoding = ENC_LITTLE_ENDIAN,
			custom_decode = function(buffer, pinfo, tree_item)
				local ts = buffer(0, 4)
				local env = buffer(4, 4)
				tree_item:add(env, string.format("Environment: %s", string.reverse(env:string())))
				tree_item:add(ts, string.format("Timestamp: %s", format_date(ts:le_uint())))
			end,
		},
		sender_comp = {
			length = 8,
			abbr = "txse_rake_tcp.sender_comp",
			name = "SenderComp",
			ftype = ftypes.STRING,
			encoding = ENC_ASCII,
		},
		token = {
			length = 8,
			abbr = "txse_rake_tcp.token",
			name = "Token",
			ftype = ftypes.STRING,
			encoding = ENC_ASCII,
		},
		next_seq = {
			length = 8,
			abbr = "txse_rake_tcp.next_seq",
			name = "Next Sequence Number",
			ftype = ftypes.INT64,
			encoding = ENC_LITTLE_ENDIAN,
		},
		highest_seq = {
			length = 8,
			abbr = "txse_rake_tcp.highest_seq",
			name = "Highest Known Sequence",
			ftype = ftypes.INT64,
			encoding = ENC_LITTLE_ENDIAN,
		},
		response_code = {
			length = 1,
			abbr = "txse_rake_tcp.response_code",
			name = "Response Code",
			ftype = ftypes.INT8,
			encoding = ENC_NA,
			value_map = {
				[0] = "SUCCESS",
				[1] = "INCORRECT_SENDER_COMP",
				[2] = "INCORRECT_SESSION",
				[3] = "INVALID_NEXT_SEQUENCE",
				[4] = "INVALID_CONFIGURATION",
				[5] = "INCORRECT_TOKEN",
			},
		},
		num_stream_ids = {
			length = 1,
			abbr = "txse_rake_tcp.num_stream_ids",
			name = "Number of Stream IDs",
			ftype = ftypes.INT8,
			encoding = ENC_NA,
		},
		instance = {
			length = 4,
			abbr = "txse_rake_tcp.instance",
			name = "Instance",
			ftype = ftypes.INT32,
			encoding = ENC_LITTLE_ENDIAN,
		},
		stream_id = {
			length = 1,
			abbr = "txse_rake_tcp.stream_id",
			name = "Stream ID",
			ftype = ftypes.UINT8,
			encoding = ENC_NA,
		},
		unsequenced_payload = {
			abbr = "txse_rake_tcp.unsequenced_payload",
			name = "Unsequenced Payload",
			ftype = ftypes.BYTES,
			encoding = ENC_NA,
			custom_decode = function(buffer, pinfo, tree_item)
				txse_seed_unseq():call(buffer:tvb(), pinfo, tree_item)
			end
		},
		debug_message = {
			abbr = "txse_rake_tcp.debug_message",
			name = "Debug Message",
			ftype = ftypes.STRING,
			encoding = ENC_ASCII,
		},
		sequenced_payload = {
			abbr = "txse_rake_tcp.sequenced_payload",
			name = "Sequenced Payload",
			ftype = ftypes.BYTES,
			encoding = ENC_NA,
			custom_decode = function(buffer, pinfo, tree_item)
				txse_seed_seq():call(buffer:tvb(), pinfo, tree_item)
			end
		},
	},
	messages = {
		{
			name = "LogonRequest",
			type = "5",
			fields = {
				"msg_length",
				"msg_type",
				"rake_session",
				"sender_comp",
				"token",
				"next_seq",
			},
		},
		{
			name = "LogonResponse",
			type = "1",
			fields = {
				"msg_length",
				"msg_type",
				"rake_session",
				"next_seq",
				"highest_seq",
				"response_code",
				"num_stream_ids",
				"instance",
			},
		},
		{
			name = "MemberHeartbeat",
			type = "7",
			fields = {
				"msg_length",
				"msg_type",
			},
		},
		{
			name = "UnsequencedMessage",
			type = "6",
			fields = {
				"msg_length",
				"msg_type",
				"unsequenced_payload",
			},
		},
		{
			name = "Debug",
			type = "0",
			fields = {
				"msg_length",
				"msg_type",
				"debug_message",
			},
		},
		{
			name = "EndOfSession",
			type = "4",
			fields = {
				"msg_length",
				"msg_type",
			},
		},
		{
			name = "SequencedMessage",
			type = "2",
			fields = {
				"msg_length",
				"msg_type",
				"stream_id",
				"sequenced_payload",
			},
		},
		{
			name = "ServerHeartbeat",
			type = "3",
			fields = {
				"msg_length",
				"msg_type",
			},
		},
	},
}

local txse_rake_tcp, messages = TXSE.Proto.new(RakeTcpSpec)

function txse_rake_tcp.dissector(buffer, pinfo, tree)
	local length = buffer:len()
	if length < 2 then
		-- Don't even have a two-byte length? Doesn't look like RAKE.
		return 0
	end

	local offset = 0
	while length - offset > 2 do
		local length_buffer = buffer(offset, 2)
		local msg_length = length_buffer:le_uint()

		local bytes_available_now = length - offset
		local bytes_needed_now = msg_length + 2
		if bytes_available_now < bytes_needed_now then
			-- TCP segment stitching into the next segment and we know exactly how many bytes required
			pinfo.desegment_len = offset
			pinfo.desegment_len = bytes_needed_now - bytes_available_now
			return length
		end

		local msg_type_buffer = buffer(offset + 2, 1)
		local msg_type = msg_type_buffer:uint()
		local msg = messages.mapping[msg_type]
		local msg_buffer = buffer(offset, bytes_needed_now)

		pinfo.cols.protocol = txse_rake_tcp.name
		if msg then
			pinfo.cols.info = msg.name
			local subtree = tree:add(txse_rake_tcp, msg_buffer, string.format("TXSE RAKE TCP %s", msg.name))
			msg:parse(msg_buffer, pinfo, subtree)
		else
			pinfo.cols.info = "Unknown RAKE message type"
			tree:add(txse_rake_tcp, msg_buffer, string.format("TXSE RAKE TCP Unknown Message Type '%c'", msg_type))
		end

		offset = offset + bytes_needed_now
	end

	if offset == length then
		-- Consumed the entire buffer
		return length
	elseif length - offset == 1 then
		-- Need at least one more byte to get the length, but cannot tell how many more bytes yet
		pinfo.desegement_offset = offset
		pinfo.desegment_len = DESEGMENT_ONE_MORE_SEGMENT
	else
		assert(false)
		return 0
	end
end

txse_rake_tcp:register_heuristic("tcp", function(buffer, pinfo, tree)
	-- Must be long enough to have the two-byte length and msg_type field
	if buffer:len() < 3 then
		return false
	end

	local msg_type = buffer(2, 1):uint()
	local msg = messages.mapping[msg_type]

	if not msg then
		-- Is not a known message type
		return false
	end

	local msg_len = buffer(0, 2):le_uint() + 2
	if msg_len < 3 or msg_len > 1500 then
		-- Doesn't seem to be a "reasonable" size embedded in the message
		return false
	end

	if (msg_len + 2) < msg.min_size then
		-- The message length is not large enough to be the specifieid message type
		return false
	end

	return txse_rake_tcp.dissector(buffer, pinfo, tree) ~= 0
end)

DissectorTable.get("tcp.port"):add_for_decode_as(txse_rake_tcp)
