local TXSE = require("txse-lib")

local SEED = require("txse-seed")

local SeedSeqSpec = {
	name = "txse_seed_seq",
	description = "TXSE SEED Binary Order Entry Protocol v0.8 (Sequenced)",
	fields = SEED.fields,
	messages = {
		{
			name = "TradingSessionStatus",
			type = "i",
			fields = {
				"msg_type",
				"trading_session_status_presence",
				"transact_time",
				"market_hours_state",
				"session_trading_state",
			},
			presence_fields = {
				"operational_halt_reason",
				"regulatory_halt_reason",
			},
		},
		{
			name = "DefineSymbol",
			type = "s",
			fields = {
				"msg_type",
				"transact_time",
				"symbol_id",
				"symbol_root",
				"symbol_suffix",
				"matching_engine_id",
				"define_symbol_bit_fields",
				"lot_size",
			},
		},
		{
			name = "SymbolStatus",
			type = "y",
			fields = {
				"msg_type",
				"symbol_status_presence",
				"transact_time",
				"symbol_trading_state",
				"ssr_state",
			},
			presence_fields = {
				"operational_halt_reason",
				"regulatory_halt_reason",
			},
		},
		{
			name = "LimitOrderAccepted",
			type = "I",
			fields = {
				"msg_type",
				"limit_order_presence",
				"transact_time",
				"order_id",
				"clordid",
				"quantity",
				"limit_order_bit_fields",
				"symbol_id",
				"price",
			},
			presence_fields = {
				"smp_scope",
				"smp_instruction",
				"price_slide_instruction",
				"min_qty",
				"max_floor_qty",
				"max_replenish_qty_range",
				"max_replenish_time_range",
				"reference_price_target",
				"expire_time",
				"user_data",
				"mpid",
				"member_group",
				"locate_broker",
			},
		},
		{
			name = "LimitOrderRejected",
			type = "U",
			fields = {
				"msg_type",
				"limit_order_presence",
				"transact_time",
				"clordid",
				"quantity",
				"limit_order_bit_fields",
				"symbol_id",
				"price",
				"reject_reason",
			},
			presence_fields = {
				"smp_scope",
				"smp_instruction",
				"price_slide_instruction",
				"min_qty",
				"max_floor_qty",
				"max_replenish_qty_range",
				"max_replenish_time_range",
				"reference_price_target",
				"expire_time",
				"user_data",
				"mpid",
				"member_group",
				"locate_broker",
			},
		},
		{
			name = "MarketOrderAccepted",
			type = "D",
			fields = {
				"msg_type",
				"market_order_presence",
				"transact_time",
				"order_id",
				"clordid",
				"quantity",
				"market_order_bit_fields",
				"symbol_id",
			},
			presence_fields = {
				"smp_scope",
				"smp_instruction",
				"user_data",
				"mpid",
				"member_group",
				"locate_broker",
			},
		},
		{
			name = "MarketOrderRejected",
			type = "T",
			fields = {
				"msg_type",
				"market_order_presence",
				"transact_time",
				"clordid",
				"quantity",
				"market_order_bit_fields",
				"symbol_id",
				"reject_reason",
			},
			presence_fields = {
				"smp_scope",
				"smp_instruction",
				"user_data",
				"mpid",
				"member_group",
				"locate_broker",
			},
		},
		{
			name = "OrderCanceled",
			type = "X",
			fields = {
				"msg_type",
				"transact_time",
				"order_id",
				"orig_clordid",
				"cancel_reason",
			},
		},
		{
			name = "CancelRejected",
			type = "W",
			fields = {
				"msg_type",
				"transact_time",
				"orig_clordid",
				"reject_reason",
			},
		},
		{
			name = "OrderExecuted",
			type = "E",
			fields = {
				"msg_type",
				"transact_time",
				"order_id",
				"orig_clordid",
				"price",
				"exec_id",
				"exec_quantity",
				"leaves_quantity",
			},
		},
		{
			name = "OrderModified",
			type = "Y",
			fields = {
				"msg_type",
				"modify_order_presence",
				"transact_time",
				"order_id",
				"clordid",
				"orig_clordid",
				"leaves_quantity",
			},
			presence_fields = {
				"quantity",
				"modify_order_bit_fields",
				"locate_broker",
			},
		},
		{
			name = "ModifyRejected",
			type = "N",
			fields = {
				"msg_type",
				"modify_order_presence",
				"transact_time",
				"clordid",
				"orig_clordid",
				"reject_reason",
			},
		},
		{
			name = "OrderReplaced",
			type = "J",
			fields = {
				"msg_type",
				"replace_order_presence",
				"transact_time",
				"order_id",
				"clordid",
				"orig_clordid",
				"replace_order_bit_fields",
				"leaves_quantity",
			},
			presence_fields = {
				"price",
				"quantity",
				"max_floor_qty",
				"locate_broker",
				"rank_price",
				"display_price",
			},
		},
		{
			name = "ReplaceRejected",
			type = "K",
			fields = {
				"msg_type",
				"replace_order_presence",
				"transact_time",
				"clordid",
				"orig_clordid",
				"replace_order_bit_fields",
				"reject_reason",
			},
			presence_fields = {
				"price",
				"quantity",
				"max_floor_qty",
				"locate_broker",
			},
		},
		{
			name = "OrderRestated",
			type = "F",
			fields = {
				"msg_type",
				"restated_presence",
				"transact_time",
				"order_id",
				"clordid",
				"restatement_reason",
			},
			presence_fields = {
				"rank_price",
				"display_price",
				"quantity",
			},
		},
		{
			name = "MassCancelAccepted",
			type = "O",
			fields = {
				"msg_type",
				"mass_cancel_presence",
				"transact_time",
				"mass_cancel_scope",
				"mass_cancel_id",
			},
			presence_fields = {
				"mpid",
				"sender_comp",
				"member_group",
				"clordid",
			},
		},
		{
			name = "MassCancelRejected",
			type = "P",
			fields = {
				"msg_type",
				"mass_cancel_presence",
				"transact_time",
				"mass_cancel_request_id",
				"mass_cancel_scope",
				"reject_reason",
			},
			presence_fields = {
				"mpid",
				"sender_comp",
				"member_group",
				"clordid",
			},
		},
		{
			name = "MassCancelResult",
			type = "Q",
			fields = {
				"msg_type",
				"transact_time",
				"mass_cancel_request_id",
				"mass_cancel_id",
				"canceled_count",
			},
		},
	},
}

local txse_seed_seq, messages = TXSE.Proto.new(SeedSeqSpec)

function txse_seed_seq.dissector(buffer, pinfo, tree)
	local length = buffer:len()
	if length < 1 then
		-- Don't even have a one-byte message type? Doesn't look like SEED.
		return 0
	end

	pinfo.cols.protocol = txse_seed_seq.name

	local msg_type = buffer(0, 1):uint()
	local msg = messages.mapping[msg_type]

	if msg then
		pinfo.cols.info = msg.name
		local subtree = tree:add(txse_seed_seq, buffer, string.format("TXSE SEED SEQ %s", msg.name))
		msg:parse(buffer, pinfo, subtree)
	else
		pinfo.cols.info = string.format("Unknown SEED SEQ message type '%c'", msg_type)
		tree:add(txse_seed_seq, buffer, string.format("TXSE SEED SEQ Unknown Message Type '%c'", msg_type))
	end

	return buffer:len()
end
