local TXSE = require("txse-lib")

local SEED = require("txse-seed")

local SeedUnseqSpec = {
	name = "txse_seed_unseq",
	description = "TXSE SEED Binary Order Entry Protocol v0.8 (Unsequenced)",
	fields = SEED.fields,
	messages = {
		{
			name = "LimitOrder",
			type = "L",
			fields = {
				"msg_type",
				"limit_order_presence",
				"clordid",
				"quantity",
				"limit_order_bit_fields",
				"symbol_id",
				"price",
			},
			presence_fields = {
				"smp_scope",
				"smp_instruction",
				"price_slide_instruction",
				"min_qty",
				"max_floor_qty",
				"max_replenish_qty_range",
				"max_replenish_time_range",
				"reference_price_target",
				"expire_time",
				"user_data",
				"mpid",
				"member_group",
				"locate_broker",
			},
		},
		{
			name = "MarketOrder",
			type = "A",
			fields = {
				"msg_type",
				"market_order_presence",
				"clordid",
				"quantity",
				"market_order_bit_fields",
				"symbol_id",
			},
			presence_fields = {
				"smp_scope",
				"smp_instruction",
				"user_data",
				"mpid",
				"member_group",
				"locate_broker",
			},
		},
		{
			name = "CancelOrder",
			type = "C",
			fields = {
				"msg_type",
				"orig_clordid",
			},
		},
		{
			name = "ModifyOrder",
			type = "M",
			fields = {
				"msg_type",
				"modify_order_presence",
				"clordid",
				"orig_clordid",
			},
			presence_fields = {
				"quantity",
				"modify_order_bit_fields",
				"locate_broker",
			},
		},
		{
			name = "ReplaceOrder",
			type = "R",
			fields = {
				"msg_type",
				"replace_order_presence",
				"clordid",
				"orig_clordid",
				"replace_order_bit_fields",
			},
			presence_fields = {
				"price",
				"quantity",
				"max_floor_qty",
				"locate_broker",
			},
		},
		{
			name = "MassCancel",
			type = "V",
			fields = {
				"msg_type",
				"mass_cancel_presence",
				"mass_cancel_request_id",
				"mass_cancel_scope",
			},
			presence_fields = {
				"mpid",
				"sender_comp",
				"member_group",
				"clordid",
			}
		},
	},
}

local txse_seed_unseq, messages = TXSE.Proto.new(SeedUnseqSpec)

function txse_seed_unseq.dissector(buffer, pinfo, tree)
	local length = buffer:len()
	if length < 1 then
		-- Don't even have a one-byte message type? Doesn't look like SEED.
		return 0
	end

	pinfo.cols.protocol = txse_seed_unseq.name

	local msg_type = buffer(0, 1):uint()
	local msg = messages.mapping[msg_type]

	if msg then
		pinfo.cols.info = msg.name
		local subtree = tree:add(txse_seed_unseq, buffer, string.format("TXSE SEED UNSEQ %s", msg.name))
		msg:parse(buffer, pinfo, subtree)
	else
		pinfo.cols.info = string.format("Unknown SEED UNSEQ message type '%c'", msg_type)
		local subtree = tree:add(txse_seed_unseq, buffer,
			string.format("TXSE SEED UNSEQ Unknown Message Type '%c'", msg_type))
	end

	return buffer:len()
end
