<?php

namespace MrB\PromoBadge\Admin;

use MrB\PromoBadge\Repository\Promo_Set_Option_Repository;

class Admin_Service_Provider {

    protected $repo;

    public function __construct( Promo_Set_Option_Repository $repo ) {
        $this->repo = $repo;
    }

    public function register() {
        add_action( 'admin_menu', [ $this, 'register_menu' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );
    }

    public function register_menu() {
        add_submenu_page(
            'woocommerce',
            __( 'Promo Badge', 'mrb-promo-badge' ),
            __( 'Promo Badge', 'mrb-promo-badge' ),
            'manage_woocommerce',
            'mrb-promo-badge',
            [ $this, 'render_page' ]
        );
    }

    public function enqueue_assets( $hook ) {
        if ( $hook !== 'woocommerce_page_mrb-promo-badge' ) {
            return;
        }

        wp_enqueue_style( 'wp-color-picker' );

        wp_enqueue_style(
            'mrb-pb-admin',
            MRB_PB_URL . 'assets/css/admin.css',
            [],
            MRB_PB_VERSION
        );

        wp_enqueue_script(
            'mrb-pb-admin',
            MRB_PB_URL . 'assets/js/admin.js',
            [ 'jquery', 'wp-color-picker' ],
            MRB_PB_VERSION,
            true
        );

        wp_localize_script( 'mrb-pb-admin', 'MRBPBAdmin', [
            'mediaTitle'         => __( 'Select badge image', 'mrb-promo-badge' ),
            'mediaButton'        => __( 'Use image', 'mrb-promo-badge' ),
            'bulkActionRequired' => __( 'Please select a bulk action.', 'mrb-promo-badge' ),
            'noBadgesSelected'   => __( 'Please select at least one badge.', 'mrb-promo-badge' ),
        ] );

        wp_enqueue_media();
    }

    public function render_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        if ( isset( $_POST['mrb_pb_action'], $_POST['mrb_pb_nonce'] ) && wp_verify_nonce( wp_unslash( $_POST['mrb_pb_nonce'] ), 'mrb_pb_save_badge' ) ) {
            $action = sanitize_text_field( wp_unslash( $_POST['mrb_pb_action'] ) );

            if ( $action === 'save_badge' ) {
                $this->handle_save_badge();
            } elseif ( $action === 'delete_badge' ) {
                $this->handle_delete_badge();
            } elseif ( $action === 'duplicate_badge' ) {
                $this->handle_duplicate_badge();
            } elseif ( $action === 'bulk_enable' || $action === 'bulk_disable' ) {
                $this->handle_bulk_toggle();
            }
        }

        settings_errors( 'mrb_pb' );

        $current_action = isset( $_GET['action'] ) ? sanitize_key( $_GET['action'] ) : 'list';

        if ( $current_action === 'new' || $current_action === 'edit' ) {
            $badge_id = isset( $_GET['badge'] ) ? sanitize_text_field( wp_unslash( $_GET['badge'] ) ) : '';
            $badge    = $current_action === 'edit'
                ? $this->repo->get_badge_config( $badge_id )
                : null;

            if ( ! $badge ) {
                $badge = Promo_Set_Option_Repository::get_default_badge_template();
            }

            include MRB_PB_PATH . 'templates/admin/badge-editor.php';

            return;
        }

        $badges = $this->repo->get_badge_configs();

        include MRB_PB_PATH . 'templates/admin/badge-list.php';
    }

    protected function handle_save_badge() {
        $raw   = isset( $_POST['mrb_pb'] ) ? wp_unslash( $_POST['mrb_pb'] ) : [];
        $badge = $this->sanitize_badge( $raw );

        $this->repo->save_badge_config( $badge );

        add_settings_error( 'mrb_pb', 'mrb_pb_saved', __( 'Badge saved.', 'mrb-promo-badge' ), 'updated' );

        // Redirect back to edit page if editing, otherwise to list
        $badge_id = $badge['id'] ?? '';
        if ( $badge_id ) {
            wp_safe_redirect( admin_url( 'admin.php?page=mrb-promo-badge&action=edit&badge=' . urlencode( $badge_id ) ) );
        } else {
            wp_safe_redirect( admin_url( 'admin.php?page=mrb-promo-badge' ) );
        }
        exit;
    }

    protected function handle_delete_badge() {
        $badge_id = isset( $_POST['badge_id'] ) ? sanitize_text_field( wp_unslash( $_POST['badge_id'] ) ) : '';

        if ( $badge_id ) {
            $this->repo->delete_badge( $badge_id );
            add_settings_error( 'mrb_pb', 'mrb_pb_deleted', __( 'Badge deleted.', 'mrb-promo-badge' ), 'updated' );
        }

        wp_safe_redirect( admin_url( 'admin.php?page=mrb-promo-badge' ) );
        exit;
    }

    protected function sanitize_badge( array $data ) : array {
        $defaults = Promo_Set_Option_Repository::get_default_badge_template();

        // Check if responsive checkboxes were sent (they might be missing if unchecked)
        $show_on_desktop = isset( $data['responsive']['show_on_desktop'] ) && ! empty( $data['responsive']['show_on_desktop'] );
        $show_on_mobile  = isset( $data['responsive']['show_on_mobile'] ) && ! empty( $data['responsive']['show_on_mobile'] );

        $sanitized = [
            'id'        => sanitize_text_field( $data['id'] ?? '' ),
            'enabled'   => ! empty( $data['enabled'] ),
            'name'      => sanitize_text_field( $data['name'] ?? $defaults['name'] ),
            'priority'  => intval( $data['priority'] ?? $defaults['priority'] ),
            'type'      => in_array( $data['type'] ?? 'text', [ 'text', 'image' ], true ) ? $data['type'] : 'text',
            'placement' => in_array( $data['placement'] ?? 'top-left', [ 'top-left', 'top-right', 'bottom-left', 'bottom-right' ], true )
                ? $data['placement']
                : 'top-left',
            'responsive' => [
                'show_on_desktop' => $show_on_desktop,
                'show_on_mobile'  => $show_on_mobile,
                'mobile_scale'    => $this->sanitize_decimal( $data['responsive']['mobile_scale'] ?? $defaults['responsive']['mobile_scale'] ),
                'custom_css_desktop' => isset( $data['responsive']['custom_css_desktop'] ) ? $this->sanitize_css( $data['responsive']['custom_css_desktop'] ) : '',
                'custom_css_tablet'  => isset( $data['responsive']['custom_css_tablet'] ) ? $this->sanitize_css( $data['responsive']['custom_css_tablet'] ) : '',
                'custom_css_mobile'  => isset( $data['responsive']['custom_css_mobile'] ) ? $this->sanitize_css( $data['responsive']['custom_css_mobile'] ) : '',
            ],
            'design' => [
                'text'        => sanitize_text_field( $data['design']['text'] ?? $defaults['design']['text'] ),
                'text_color'  => sanitize_hex_color( $data['design']['text_color'] ?? '#ffffff' ) ?: '#ffffff',
                'font_family' => sanitize_text_field( $data['design']['font_family'] ?? '' ),
                'font_weight' => sanitize_text_field( $data['design']['font_weight'] ?? '600' ),
                'font_size'   => sanitize_text_field( $data['design']['font_size'] ?? '14px' ),
                'image_id'    => absint( $data['design']['image_id'] ?? 0 ),
                'image_url'   => esc_url_raw( $data['design']['image_url'] ?? '' ),
                'width'       => sanitize_text_field( $data['design']['width'] ?? '72px' ),
                'height'      => sanitize_text_field( $data['design']['height'] ?? '72px' ),
                'object_fit'  => sanitize_text_field( $data['design']['object_fit'] ?? 'contain' ),
                'styles'      => [
                    'background_color' => $this->sanitize_background_color( $data['design']['styles']['background_color'] ?? '' ),
                    'use_background'   => isset( $data['design']['styles']['use_background'] ) && ! empty( $data['design']['styles']['use_background'] ),
                    'border'           => sanitize_text_field( $data['design']['styles']['border'] ?? '' ),
                    'border_radius'    => sanitize_text_field( $data['design']['styles']['border_radius'] ?? '999px' ),
                    'padding'          => sanitize_text_field( $data['design']['styles']['padding'] ?? '0.45rem 0.9rem' ),
                    'offset_x'         => $this->sanitize_decimal( $data['design']['styles']['offset_x'] ?? 12 ),
                    'offset_y'         => $this->sanitize_decimal( $data['design']['styles']['offset_y'] ?? 12 ),
                    'z_index'          => intval( $data['design']['styles']['z_index'] ?? 15 ),
                ],
            ],
            'rules' => [
                'on_sale'      => isset( $data['rules']['on_sale'] ) && ! empty( $data['rules']['on_sale'] ),
                'category_ids' => $this->sanitize_array_ids( $data['rules']['category_ids'] ?? null ),
                'tag_ids'      => $this->sanitize_array_ids( $data['rules']['tag_ids'] ?? null ),
                'brand_ids'    => $this->sanitize_array_ids( $data['rules']['brand_ids'] ?? null ),
                'logic'        => in_array( $data['rules']['logic'] ?? 'or', [ 'or', 'and' ], true ) ? $data['rules']['logic'] : 'or',
            ],
        ];

        return $sanitized;
    }

    protected function sanitize_decimal( $value ) : float {
        return (float) str_replace( ',', '.', (string) $value );
    }

    protected function sanitize_array_ids( $value ) : array {
        // Handle case where multi-select sends empty string or null
        if ( empty( $value ) || $value === '' ) {
            return [];
        }

        if ( ! is_array( $value ) ) {
            // Single value - convert to array
            $value = [ $value ];
        }

        return array_filter( array_map( 'absint', $value ) );
    }

    protected function sanitize_background_color( $value ) : string {
        $value = trim( (string) $value );

        if ( empty( $value ) ) {
            return '';
        }

        $sanitized = sanitize_hex_color( $value );

        return $sanitized ?: '';
    }

    /**
     * Sanitize CSS input - allows safe CSS properties while preventing XSS
     * 
     * @param string|mixed $css Raw CSS input
     * @return string Sanitized CSS
     */
    protected function sanitize_css( $css ) : string {
        // Convert to string if not already, handle null
        if ( $css === null ) {
            return '';
        }
        
        if ( ! is_string( $css ) ) {
            $css = (string) $css;
        }
        
        // Return empty string if empty
        $css = trim( $css );
        if ( empty( $css ) ) {
            return '';
        }

        // Use sanitize_textarea_field to handle basic sanitization
        // This preserves line breaks and whitespace which are important for CSS
        $css = sanitize_textarea_field( $css );

        // Remove any script tags and javascript: protocols
        // Note: preg_replace in PHP doesn't use 'g' flag (that's JavaScript), it replaces all by default
        $css = preg_replace( '/<script\b[^<]*(?:(?!<\/script>)<[^<]*)*<\/script>/i', '', $css );
        $css = preg_replace( '/javascript:/i', '', $css );
        $css = preg_replace( '/on\w+\s*=/i', '', $css );

        // Remove potentially dangerous CSS expressions
        $css = preg_replace( '/expression\s*\(/i', '', $css );
        $css = preg_replace( '/behavior\s*:/i', '', $css );
        $css = preg_replace( '/-moz-binding/i', '', $css );

        // Remove @import and @charset which could be used for XSS
        $css = preg_replace( '/@import/i', '', $css );
        $css = preg_replace( '/@charset/i', '', $css );

        // Remove any HTML tags (but preserve CSS structure)
        // wp_strip_all_tags removes HTML but preserves text content
        $css = wp_strip_all_tags( $css );

        // Return trimmed CSS (but preserve internal whitespace)
        return trim( $css );
    }

    protected function handle_duplicate_badge() {
        $badge_id = isset( $_POST['badge_id'] ) ? sanitize_text_field( wp_unslash( $_POST['badge_id'] ) ) : '';

        if ( ! $badge_id ) {
            add_settings_error( 'mrb_pb', 'mrb_pb_error', __( 'Badge ID is required.', 'mrb-promo-badge' ), 'error' );
            wp_safe_redirect( admin_url( 'admin.php?page=mrb-promo-badge' ) );
            exit;
        }

        $original = $this->repo->get_badge_config( $badge_id );

        if ( ! $original ) {
            add_settings_error( 'mrb_pb', 'mrb_pb_error', __( 'Badge not found.', 'mrb-promo-badge' ), 'error' );
            wp_safe_redirect( admin_url( 'admin.php?page=mrb-promo-badge' ) );
            exit;
        }

        // Create duplicate with new ID
        $duplicate = $original;
        $duplicate['id']      = '';
        $duplicate['name']    = $original['name'] . ' ' . __( '(Copy)', 'mrb-promo-badge' );
        $duplicate['enabled'] = false; // Disable by default

        $this->repo->save_badge_config( $duplicate );

        add_settings_error( 'mrb_pb', 'mrb_pb_duplicated', __( 'Badge duplicated.', 'mrb-promo-badge' ), 'updated' );

        wp_safe_redirect( admin_url( 'admin.php?page=mrb-promo-badge' ) );
        exit;
    }

    protected function handle_bulk_toggle() {
        $action = sanitize_text_field( wp_unslash( $_POST['mrb_pb_action'] ) );
        $badge_ids = isset( $_POST['badge_ids'] ) ? array_map( 'sanitize_text_field', (array) $_POST['badge_ids'] ) : [];

        if ( empty( $badge_ids ) ) {
            add_settings_error( 'mrb_pb', 'mrb_pb_error', __( 'No badges selected.', 'mrb-promo-badge' ), 'error' );
            wp_safe_redirect( admin_url( 'admin.php?page=mrb-promo-badge' ) );
            exit;
        }

        $enabled = $action === 'bulk_enable';
        $count   = 0;

        foreach ( $badge_ids as $badge_id ) {
            $badge = $this->repo->get_badge_config( $badge_id );

            if ( $badge ) {
                $badge['enabled'] = $enabled;
                $this->repo->save_badge_config( $badge );
                $count++;
            }
        }

        $message = $enabled
            ? sprintf( _n( '%d badge enabled.', '%d badges enabled.', $count, 'mrb-promo-badge' ), $count )
            : sprintf( _n( '%d badge disabled.', '%d badges disabled.', $count, 'mrb-promo-badge' ), $count );

        add_settings_error( 'mrb_pb', 'mrb_pb_bulk_toggled', $message, 'updated' );

        wp_safe_redirect( admin_url( 'admin.php?page=mrb-promo-badge' ) );
        exit;
    }
}

