<?php

namespace MrB\PromoBadge\Frontend;

use MrB\PromoBadge\Model\Promo_Set;
use MrB\PromoBadge\Model\Text_Badge_Design;
use MrB\PromoBadge\Model\Image_Badge_Design;

class Badge_Renderer {

    public function render( Promo_Set $set, $context ) : string {
        $design     = $set->get_design();
        $placement  = $set->get_placement()->get_position();
        $responsive = $set->get_responsive();

        $classes   = [
            'mrb-promo-badge',
            'mrb-promo-badge--' . esc_attr( $placement ),
            'mrb-promo-badge--type-' . esc_attr( $design->get_type() ),
        ];

        // Add data attributes for responsive settings
        $data_attrs = [];
        $data_attrs['data-badge-id'] = esc_attr( $set->get_id() );
        $data_attrs['data-mobile-scale'] = esc_attr( $responsive->get_mobile_scale() );
        $data_attrs['data-show-mobile'] = $responsive->should_show_on( 'mobile' ) ? '1' : '0';
        $data_attrs['data-show-desktop'] = $responsive->should_show_on( 'desktop' ) ? '1' : '0';

        $style_parts   = [];
        $offset_x      = $design->get_offset_x();
        $offset_y      = $design->get_offset_y();

        // basic pozicioniranje
        switch ( $placement ) {
            case 'top-left':
                $style_parts[] = "top: {$offset_y}px;";
                $style_parts[] = "left: {$offset_x}px;";
                break;
            case 'top-right':
                $style_parts[] = "top: {$offset_y}px;";
                $style_parts[] = "right: {$offset_x}px;";
                break;
            case 'bottom-left':
                $style_parts[] = "bottom: {$offset_y}px;";
                $style_parts[] = "left: {$offset_x}px;";
                break;
            case 'bottom-right':
                $style_parts[] = "bottom: {$offset_y}px;";
                $style_parts[] = "right: {$offset_x}px;";
                break;
        }

        // Background color - only apply if explicitly enabled
        if ( $design->get_use_background() && $design->get_background_color() ) {
            $style_parts[] = 'background:' . esc_attr( $design->get_background_color() ) . ';';
        }
        if ( $design->get_border() ) {
            $style_parts[] = 'border:' . esc_attr( $design->get_border() ) . ';';
        }
        if ( $design->get_border_radius() ) {
            $style_parts[] = 'border-radius:' . esc_attr( $design->get_border_radius() ) . ';';
        }
        if ( $design->get_padding() ) {
            $style_parts[] = 'padding:' . esc_attr( $design->get_padding() ) . ';';
        }

        $style_parts[] = 'position:absolute;';
        $style_parts[] = 'z-index:' . (int) $design->get_z_index() . ';';

        // Mobile scale is handled by JavaScript for better device detection

        $inner_html = '';

        if ( $design instanceof Text_Badge_Design ) {
            $text_style = [];
            $text_style[] = 'color:' . esc_attr( $design->get_text_color() ) . ';';
            if ( $design->get_font_family() ) {
                $text_style[] = 'font-family:' . esc_attr( $design->get_font_family() ) . ';';
            }
            if ( $design->get_font_weight() ) {
                $text_style[] = 'font-weight:' . esc_attr( $design->get_font_weight() ) . ';';
            }
            if ( $design->get_font_size() ) {
                $text_style[] = 'font-size:' . esc_attr( $design->get_font_size() ) . ';';
            }

            $inner_html = sprintf(
                '<span class="mrb-promo-badge__text" style="%s">%s</span>',
                esc_attr( implode( '', $text_style ) ),
                esc_html( $design->get_text() )
            );
        } elseif ( $design instanceof Image_Badge_Design ) {
            $image_id = $design->get_image_id();
            $src      = $image_id ? wp_get_attachment_image_url( $image_id, 'full' ) : '';

            if ( ! $src && $design->get_image_url() ) {
                $src = $design->get_image_url();
            }

            if ( $src ) {
                $img_style = [];

                if ( $design->get_width() ) {
                    $img_style[] = 'width:' . esc_attr( $design->get_width() ) . ';';
                }
                if ( $design->get_height() ) {
                    $img_style[] = 'height:' . esc_attr( $design->get_height() ) . ';';
                }
                $img_style[] = 'object-fit:' . esc_attr( $design->get_object_fit() ) . ';';

                $inner_html = sprintf(
                    '<img class="mrb-promo-badge__img" src="%s" alt="" style="%s" />',
                    esc_url( $src ),
                    esc_attr( implode( '', $img_style ) )
                );
            }
        }

        if ( ! $inner_html ) {
            return '';
        }

        // Build data attributes string
        $data_string = '';
        foreach ( $data_attrs as $key => $value ) {
            $data_string .= ' ' . esc_attr( $key ) . '="' . esc_attr( $value ) . '"';
        }

        return sprintf(
            '<div class="%s" style="%s"%s>%s</div>',
            esc_attr( implode( ' ', $classes ) ),
            esc_attr( implode( '', $style_parts ) ),
            $data_string,
            $inner_html
        );
    }
}
