<?php

namespace MrB\PromoBadge\Frontend;

use MrB\PromoBadge\Services\Promo_Set_Manager;
use MrB\PromoBadge\Services\Product_Context;

class Frontend_Service_Provider {

    protected $manager;

    public function __construct( Promo_Set_Manager $manager ) {
        $this->manager = $manager;
    }

    public function register() {
        add_action( 'wp', [ $this, 'maybe_hook_loop' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'maybe_enqueue_assets' ] );
        add_action( 'wp_head', [ $this, 'maybe_output_custom_css' ], 100 );
    }

    protected function is_supported_view() : bool {
        return is_shop() || is_product_taxonomy() || is_product_category() || is_product_tag() || is_search();
    }

    public function maybe_hook_loop() {
        if ( ! $this->is_supported_view() ) {
            return;
        }

        add_action( 'woocommerce_before_shop_loop_item', [ $this, 'render_badge' ], 5 );
    }

    public function maybe_enqueue_assets() {
        if ( ! $this->is_supported_view() ) {
            return;
        }

        wp_enqueue_style(
            'mrb-pb-frontend',
            MRB_PB_URL . 'assets/css/frontend.css',
            [],
            MRB_PB_VERSION
        );

        wp_enqueue_script(
            'mrb-pb-frontend',
            MRB_PB_URL . 'assets/js/frontend.js',
            [],
            MRB_PB_VERSION,
            true
        );
    }

    public function render_badge() {
        global $product;

        if ( ! $product || ! is_a( $product, 'WC_Product' ) ) {
            return;
        }

        // Better mobile detection
        $device = $this->detect_device();
        $context = new Product_Context( $product, $device, is_archive() );

        $set = $this->manager->get_matching_sets( $context, false );

        if ( ! $set ) {
            return;
        }

        echo ( new Badge_Renderer() )->render( $set, $context );
    }

    protected function detect_device() : string {
        // Try wp_is_mobile first
        if ( function_exists( 'wp_is_mobile' ) && wp_is_mobile() ) {
            return 'mobile';
        }

        // Fallback: check user agent
        if ( isset( $_SERVER['HTTP_USER_AGENT'] ) ) {
            $user_agent = strtolower( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) );
            $mobile_keywords = [ 'mobile', 'android', 'iphone', 'ipad', 'ipod', 'blackberry', 'windows phone' ];
            
            foreach ( $mobile_keywords as $keyword ) {
                if ( strpos( $user_agent, $keyword ) !== false ) {
                    return 'mobile';
                }
            }
        }

        return 'desktop';
    }

    /**
     * Output custom CSS for all enabled badges
     * CSS is scoped to badge elements and wrapped in appropriate media queries
     */
    public function maybe_output_custom_css() {
        if ( ! $this->is_supported_view() ) {
            return;
        }

        $sets = $this->manager->get_all_enabled_sets();
        
        if ( empty( $sets ) ) {
            return;
        }

        $css_parts = [];

        foreach ( $sets as $set ) {
            $responsive = $set->get_responsive();
            $badge_id = $set->get_id();
            
            // Generate unique selector for this badge using data attribute
            $selector = '.mrb-promo-badge[data-badge-id="' . esc_attr( $badge_id ) . '"]';
            
            // Desktop CSS (min-width: 769px)
            $desktop_css = $responsive->get_custom_css_desktop();
            if ( ! empty( $desktop_css ) ) {
                $css_parts[] = '@media screen and (min-width: 769px) {';
                $css_parts[] = $this->scope_css( $desktop_css, $selector );
                $css_parts[] = '}';
            }

            // Tablet CSS (481px - 768px)
            $tablet_css = $responsive->get_custom_css_tablet();
            if ( ! empty( $tablet_css ) ) {
                $css_parts[] = '@media screen and (min-width: 481px) and (max-width: 768px) {';
                $css_parts[] = $this->scope_css( $tablet_css, $selector );
                $css_parts[] = '}';
            }

            // Mobile CSS (max-width: 480px)
            $mobile_css = $responsive->get_custom_css_mobile();
            if ( ! empty( $mobile_css ) ) {
                $css_parts[] = '@media screen and (max-width: 480px) {';
                $css_parts[] = $this->scope_css( $mobile_css, $selector );
                $css_parts[] = '}';
            }
        }

        if ( ! empty( $css_parts ) ) {
            echo '<style id="mrb-pb-custom-css">' . "\n";
            // Add indentation to media query content
            $formatted = [];
            foreach ( $css_parts as $part ) {
                if ( strpos( $part, '@media' ) === 0 ) {
                    $formatted[] = $part;
                } elseif ( $part === '}' ) {
                    $formatted[] = $part;
                } else {
                    // Indent CSS rules inside media queries
                    $formatted[] = '    ' . $part;
                }
            }
            echo implode( "\n", $formatted );
            echo "\n" . '</style>' . "\n";
        }
    }

    /**
     * Scope CSS rules to the badge selector
     * Handles both direct properties and nested selectors (like .mrb-promo-badge__img)
     * Also handles modifier classes like .mrb-promo-badge--top-right
     * 
     * @param string $css Raw CSS input
     * @param string $selector Base selector to scope to (e.g., .mrb-promo-badge[data-badge-id="..."])
     * @return string Scoped CSS
     */
    protected function scope_css( string $css, string $selector ) : string {
        if ( empty( $css ) ) {
            return '';
        }

        // Remove any existing media queries (we handle those separately)
        $css = preg_replace( '/@media[^{]*\{[^}]*\}/is', '', $css );
        $css = trim( $css );
        
        // If CSS contains selectors starting with .mrb-promo-badge, scope them to our selector
        // Otherwise, treat as direct properties and wrap in selector
        if ( preg_match( '/\.mrb-promo-badge/', $css ) ) {
            // CSS already uses badge classes - we need to combine selectors properly
            // For selectors like .mrb-promo-badge--top-right, combine with our data selector
            // Result: .mrb-promo-badge[data-badge-id="..."].mrb-promo-badge--top-right
            $scoped = preg_replace_callback(
                '/\.mrb-promo-badge(?:--[\w-]+)?(?!__)/',
                function( $matches ) use ( $selector ) {
                    $full_match = $matches[0]; // e.g., ".mrb-promo-badge" or ".mrb-promo-badge--top-right"
                    // Combine: our data selector + the original badge class/modifier
                    return $selector . $full_match;
                },
                $css
            );
            return $scoped;
        } else {
            // Direct properties (no selector) - wrap in selector
            // Split by semicolons to handle multiple properties
            $properties = array_filter( array_map( 'trim', explode( ';', $css ) ) );
            if ( empty( $properties ) ) {
                return '';
            }
            
            $formatted = [];
            foreach ( $properties as $prop ) {
                if ( ! empty( $prop ) ) {
                    $formatted[] = '        ' . $prop . ';';
                }
            }
            
            return $selector . " {\n" . implode( "\n", $formatted ) . "\n    }";
        }
    }
}
