<?php

namespace MrB\PromoBadge\Repository;

use MrB\PromoBadge\Model\Image_Badge_Design;
use MrB\PromoBadge\Model\Placement;
use MrB\PromoBadge\Model\Promo_Set;
use MrB\PromoBadge\Model\Responsive_Settings;
use MrB\PromoBadge\Model\Text_Badge_Design;
use MrB\PromoBadge\Rules\Always_Rule;
use MrB\PromoBadge\Rules\Brand_Rule;
use MrB\PromoBadge\Rules\Category_Rule;
use MrB\PromoBadge\Rules\Composite_Rule;
use MrB\PromoBadge\Rules\On_Sale_Rule;
use MrB\PromoBadge\Rules\Rule_Interface;
use MrB\PromoBadge\Rules\Tag_Rule;

class Promo_Set_Option_Repository implements Promo_Set_Repository_Interface {

    const OPTION_NAME        = 'mrb_pb_badges';
    const LEGACY_OPTION_NAME = 'mrb_pb_default_badge';

    public function all_enabled() : array {
        $configs = $this->get_badge_configs();
        $sets    = [];

        foreach ( $configs as $config ) {
            if ( empty( $config['enabled'] ) ) {
                continue;
            }

            $set = $this->map_to_model( $config );

            if ( $set ) {
                $sets[] = $set;
            }
        }

        return $sets;
    }

    public function find( $id ) : ?Promo_Set {
        $config = $this->get_badge_config( $id );

        if ( ! $config || empty( $config['enabled'] ) ) {
            return null;
        }

        return $this->map_to_model( $config );
    }

    public function save( Promo_Set $set ) : int {
        return 1;
    }

    /**
     * @return array[]
     */
    public function get_badge_configs() : array {
        $option = get_option( self::OPTION_NAME, [] );

        if ( empty( $option ) ) {
            $option = $this->maybe_migrate_legacy();
        }

        $badges = $option['badges'] ?? [];

        if ( empty( $badges ) ) {
            $badges = [ $this->with_generated_id( self::get_default_badge_template() ) ];
            $this->save_badge_configs( $badges );
        }

        return array_values( $badges );
    }

    public function get_badge_config( $id ) : ?array {
        foreach ( $this->get_badge_configs() as $config ) {
            if ( (string) ( $config['id'] ?? '' ) === (string) $id ) {
                return $config;
            }
        }

        return null;
    }

    public function save_badge_configs( array $configs ) {
        update_option( self::OPTION_NAME, [
            'badges' => array_values( $configs ),
        ] );
    }

    public function save_badge_config( array $config ) {
        $configs = $this->get_badge_configs();
        $id      = $config['id'] ?? '';

        if ( ! $id ) {
            $config = $this->with_generated_id( $config );
            $id     = $config['id'];
        }

        $found = false;

        foreach ( $configs as $index => $item ) {
            if ( (string) $item['id'] === (string) $id ) {
                $configs[ $index ] = $config;
                $found             = true;
                break;
            }
        }

        if ( ! $found ) {
            $configs[] = $config;
        }

        $this->save_badge_configs( $configs );
    }

    public function delete_badge( $id ) {
        $configs = array_filter(
            $this->get_badge_configs(),
            function ( $config ) use ( $id ) {
                return (string) ( $config['id'] ?? '' ) !== (string) $id;
            }
        );

        $this->save_badge_configs( $configs );
    }

    protected function map_to_model( array $config ) : ?Promo_Set {
        $design_data = $config['design'] ?? [];
        $type        = $config['type'] ?? 'text';

        if ( $type === 'image' ) {
            $design = new Image_Badge_Design( $design_data );
        } else {
            $design = new Text_Badge_Design( $design_data );
        }

        $placement  = new Placement( $config['placement'] ?? 'top-left' );
        $responsive = new Responsive_Settings(
            ! empty( $config['responsive']['show_on_desktop'] ),
            ! empty( $config['responsive']['show_on_mobile'] ),
            (float) ( $config['responsive']['mobile_scale'] ?? 1.0 ),
            (string) ( $config['responsive']['custom_css_desktop'] ?? '' ),
            (string) ( $config['responsive']['custom_css_tablet'] ?? '' ),
            (string) ( $config['responsive']['custom_css_mobile'] ?? '' )
        );

        $rule = $this->build_rule( $config['rules'] ?? [] );

        return new Promo_Set(
            $config['id'] ?? 'default',
            $config['name'] ?? __( 'Default Promo Badge', 'mrb-promo-badge' ),
            $design,
            $rule,
            $placement,
            $responsive,
            (int) ( $config['priority'] ?? 10 ),
            ! empty( $config['enabled'] )
        );
    }

    public static function get_default_badge_template() : array {
        return [
            'id'        => '',
            'enabled'   => true,
            'name'      => __( 'Default Promo Badge', 'mrb-promo-badge' ),
            'priority'  => 10,
            'type'      => 'text',
            'placement' => 'top-left',
            'responsive' => [
                'show_on_desktop' => 1,
                'show_on_mobile'  => 1,
                'mobile_scale'    => 1,
                'custom_css_desktop' => '',
                'custom_css_tablet'  => '',
                'custom_css_mobile'  => '',
            ],
            'design'    => [
                'text'         => __( 'Promo', 'mrb-promo-badge' ),
                'text_color'   => '#ffffff',
                'font_family'  => '',
                'font_weight'  => '600',
                'font_size'    => '14px',
                'image_id'     => 0,
                'image_url'    => '',
                'width'        => '72px',
                'height'       => '72px',
                'object_fit'   => 'contain',
                'styles'       => [
                    'background_color' => '#111111',
                    'use_background'   => true,
                    'border'           => '',
                    'border_radius'    => '999px',
                    'padding'          => '0.45rem 0.9rem',
                    'offset_x'         => 12,
                    'offset_y'         => 12,
                    'z_index'          => 15,
                ],
            ],
            'rules'     => [
                'on_sale'      => false,
                'category_ids' => [],
                'tag_ids'      => [],
                'brand_ids'    => [],
                'logic'        => 'or',
            ],
        ];
    }

    protected function maybe_migrate_legacy() : array {
        $legacy = get_option( self::LEGACY_OPTION_NAME, [] );

        if ( empty( $legacy ) ) {
            return [];
        }

        $legacy = wp_parse_args( $legacy, self::get_default_badge_template() );
        $legacy = $this->with_generated_id( $legacy );

        $option = [ 'badges' => [ $legacy ] ];
        update_option( self::OPTION_NAME, $option );

        return $option;
    }

    protected function with_generated_id( array $config ) : array {
        if ( empty( $config['id'] ) ) {
            $config['id'] = uniqid( 'badge_', true );
        }

        return $config;
    }

    protected function build_rule( array $rules_config ) : Rule_Interface {
        $rule_parts = [];

        if ( ! empty( $rules_config['on_sale'] ) ) {
            $rule_parts[] = new On_Sale_Rule();
        }

        if ( ! empty( $rules_config['category_ids'] ) && is_array( $rules_config['category_ids'] ) ) {
            $category_ids = array_filter( array_map( 'absint', $rules_config['category_ids'] ) );
            if ( ! empty( $category_ids ) ) {
                $rule_parts[] = new Category_Rule( $category_ids );
            }
        }

        if ( ! empty( $rules_config['tag_ids'] ) && is_array( $rules_config['tag_ids'] ) ) {
            $tag_ids = array_filter( array_map( 'absint', $rules_config['tag_ids'] ) );
            if ( ! empty( $tag_ids ) ) {
                $rule_parts[] = new Tag_Rule( $tag_ids );
            }
        }

        if ( ! empty( $rules_config['brand_ids'] ) && is_array( $rules_config['brand_ids'] ) ) {
            $brand_ids = array_filter( array_map( 'absint', $rules_config['brand_ids'] ) );
            if ( ! empty( $brand_ids ) ) {
                $rule_parts[] = new Brand_Rule( $brand_ids );
            }
        }

        if ( empty( $rule_parts ) ) {
            return new Always_Rule();
        }

        if ( count( $rule_parts ) === 1 ) {
            return $rule_parts[0];
        }

        $logic = ( $rules_config['logic'] ?? 'or' ) === 'and' ? Composite_Rule::TYPE_AND : Composite_Rule::TYPE_OR;

        return new Composite_Rule( $rule_parts, $logic );
    }
}

