<?php

namespace MrB\PromoBadge\Update;

class Update_Checker {

	const UPDATE_SERVER_URL = 'https://the-happytail.com/mrb_promo_badge/update.php';
	const PLUGIN_SLUG = 'mrbrabus-promo-badge';
	const PLUGIN_FILE = 'mrbrabus-promo-badge/mrbrabus-promo-badge.php';

	public function __construct() {
		add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'check_for_updates' ] );
		add_filter( 'plugins_api', [ $this, 'plugin_info' ], 10, 3 );
		add_filter( 'upgrader_post_install', [ $this, 'after_install' ], 10, 3 );
	}

	/**
	 * Proverava da li postoji nova verzija
	 */
	public function check_for_updates( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$current_version = MRB_PB_VERSION;
		$remote_version = $this->get_remote_version();

		if ( $remote_version && version_compare( $current_version, $remote_version, '<' ) ) {
			$obj = new \stdClass();
			$obj->slug = self::PLUGIN_SLUG;
			$obj->plugin = self::PLUGIN_FILE;
			$obj->new_version = $remote_version;
			$obj->url = self::UPDATE_SERVER_URL;
			$obj->package = $this->get_download_url();
			$obj->requires = '5.0';
			$obj->tested = '6.4';
			$obj->requires_php = '7.4';

			$transient->response[ self::PLUGIN_FILE ] = $obj;
		}

		return $transient;
	}

	/**
	 * Vraća informacije o pluginu za "View details" modal
	 */
	public function plugin_info( $result, $action, $args ) {
		if ( $action !== 'plugin_information' ) {
			return $result;
		}

		if ( ! isset( $args->slug ) || $args->slug !== self::PLUGIN_SLUG ) {
			return $result;
		}

		$remote_info = $this->get_remote_info();

		if ( $remote_info ) {
			return $remote_info;
		}

		return $result;
	}

	/**
	 * Nakon instalacije update-a
	 */
	public function after_install( $response, $hook_extra, $result ) {
		global $wp_filesystem;

		if ( ! isset( $hook_extra['plugin'] ) || $hook_extra['plugin'] !== self::PLUGIN_FILE ) {
			return $response;
		}

		$install_directory = plugin_dir_path( dirname( dirname( __FILE__ ) ) );
		$wp_filesystem->move( $result['destination'], $install_directory );
		$result['destination'] = $install_directory;

		if ( is_plugin_active( self::PLUGIN_FILE ) ) {
			activate_plugin( self::PLUGIN_FILE );
		}

		return $result;
	}

	/**
	 * Dobija verziju sa update servera
	 */
	protected function get_remote_version() {
		$cache_key = 'mrb_pb_remote_version';
		$version = get_transient( $cache_key );

		if ( false === $version ) {
			$response = wp_remote_get(
				add_query_arg( [ 'version' => MRB_PB_VERSION ], self::UPDATE_SERVER_URL ),
				[
					'timeout' => 10,
					'sslverify' => true,
				]
			);

			if ( is_wp_error( $response ) ) {
				return false;
			}

			$body = wp_remote_retrieve_body( $response );
			$data = json_decode( $body, true );

			if ( isset( $data['new_version'] ) ) {
				$version = $data['new_version'];
				set_transient( $cache_key, $version, 12 * HOUR_IN_SECONDS );
			} else {
				return false;
			}
		}

		return $version;
	}

	/**
	 * Dobija detaljne informacije sa update servera
	 */
	protected function get_remote_info() {
		$cache_key = 'mrb_pb_remote_info';
		$info = get_transient( $cache_key );

		if ( false === $info ) {
			$response = wp_remote_get(
				add_query_arg( [ 'action' => 'plugin_information' ], self::UPDATE_SERVER_URL ),
				[
					'timeout' => 10,
					'sslverify' => true,
				]
			);

			if ( is_wp_error( $response ) ) {
				return false;
			}

			$body = wp_remote_retrieve_body( $response );
			$info = json_decode( $body, true );

			if ( $info ) {
				set_transient( $cache_key, $info, 12 * HOUR_IN_SECONDS );
			} else {
				return false;
			}
		}

		return (object) $info;
	}

	/**
	 * Vraća URL za download
	 */
	protected function get_download_url() {
		$response = wp_remote_get(
			add_query_arg( [ 'version' => MRB_PB_VERSION ], self::UPDATE_SERVER_URL ),
			[
				'timeout' => 10,
				'sslverify' => true,
			]
		);

		if ( is_wp_error( $response ) ) {
			return '';
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		return isset( $data['package'] ) ? $data['package'] : 'https://the-happytail.com/mrb_promo_badge/mrb-promo-badge.zip';
	}
}

